#include "Map.h"
#include <string.h>
#include <stdlib.h>
#include "HostMachine.h"
#include "BitmapManager.h"

#include "Tokeniser.h"

Uint16 GetEventKey(int Id1, int Id2)
{
	if(Id1 && Id2) return (Id1&255)*(Id2&255);
	if(!Id1) return Id2&65535;
	return Id1&65535;
}

void CMap::FreeStructures()
{
#define FreeList(type, name)\
	while(name)\
	{\
		type *Next = name->Next;\
		delete name;\
		name = Next;\
	}

	FreeList(Screen, ScreenList);
	FreeList(ScreenLink, ScreenLinkList);
	FreeList(MapPatch, MapPatchList);
	FreeList(DizzyPatch, DizzyPatchList);
	FreeList(ObjectDefinition, ObjectDefinitionList);
	FreeList(Object, ObjectList);
	FreeList(ObjectPatch, ObjectPatchList);
	
	int c = 65536;
	while(c--)
		FreeList(Event, EventList[c]);
}

void Object::FreePath()
{
	if(Path)
	{
		/* create a break in the list */
		Node *NPtr = Path->Next;
		Path->Next = NULL;
		FreeList(Node, NPtr);
		Path = CPtr = NULL;
	}
}

Object::~Object()
{
	FreePath();
}

#undef FreeList
/* Yawnsome constructors & destructors */
CMap::CMap()
{
	ScreenList = ScreenPtr = NULL;
	ScreenLinkList = ScreenLinkPtr = NULL;
	MapPatchList = NULL;
	DizzyPatchList = NULL;
	ObjectDefinitionList = NULL;
	ObjectList = ScreenObjectList = NULL;
	ObjectPatchList = NULL;

	int c = 65536;
	while(c--)
		EventList[c] = NULL;

	CurrentScreen = NULL;
}

CMap::~CMap()
{
	FreeStructures();
	delete CurrentScreen;
}


Screen::Screen()
{
	Next = NULL;
	ScreenName =
	ForegroundMap = 
	BackgroundMap = NULL;
	Patches = NULL;
}

Screen::~Screen()
{
	if(ScreenName) free(ScreenName);
	if(ForegroundMap) free(ForegroundMap);
	if(BackgroundMap) free(BackgroundMap);
}

ScreenLink::ScreenLink()
{
	Next = NULL;
}

MapPatch::MapPatch()
{
	Filename = NULL;
	Next = ScreenNext = NULL;
}

MapPatch::~MapPatch()
{
	if(Filename) free(Filename);
}

DizzyPatch::DizzyPatch()
{
	Next = NULL;
}

ObjectDefinition::ObjectDefinition()
{
	Next = NULL;
	Image = NULL;
	Tilemap = NULL;
	Name = NULL;
}

ObjectDefinition::~ObjectDefinition()
{
	delete Image;
	delete Tilemap;
	free(Name);
}

Object::Object()
{
	Next = ScreenNext = NULL;
	Definition = NULL;
	Path = CPtr = NULL;
	CurrentQuantum = ~0;
}

Object::Node::Node()
{
	Next = NULL;
}

ObjectPatch::ObjectPatch()
{
	Next = NULL;
}

CDialogueHandler::CDialogueHandler()
{
	Text = NULL;
	Next = NULL;
}

CDialogueHandler::~CDialogueHandler()
{
	if(Text) free(Text);
}

Event::Event()
{
	Next = NULL;
	DialogueSource = NULL;
}

Event::~Event()
{
	if(DialogueSource) free(DialogueSource);
}

/* read funcs */

#define GetInteger(v) ExpectToken(TK_NUMBER); v = Token.IQuantity
#define GetIntegerComma(v) GetInteger(v); ExpectToken(TK_COMMA)

#define GetFloat(v) ExpectToken(TK_NUMBER); v = Token.FQuantity
#define GetFloatComma(v) GetFloat(v); ExpectToken(TK_COMMA)

#define GetString(v) ExpectToken(TK_STRING); v = strdup(Token.Text)
#define GetStringComma(v) GetString(v); ExpectToken(TK_COMMA)

CBitmap *GetImage(CBitmapParent *P, bool bitmask = false)
{
	GetToken();
	switch(Token.Type)
	{
		case TK_OPENBRACE:
		{
			int NumFrames;
			GetIntegerComma(NumFrames);

			CBitmap *r;
			GetToken();
			switch(Token.Type)
			{
				default: Error = 1; return NULL;

				case TK_NUMBER:
				{
					/* regular */
					r = new CRegularBitmap(P, NumFrames, Token.IQuantity, bitmask);
					for(int c = 0; c < NumFrames; c++)
					{
						ExpectToken(TK_COMMA);
						ExpectToken(TK_STRING);
						((CRegularBitmap *)r)->SetFrameImage(c, Token.Text);
					}
				}
				break;

				case TK_STRING:
				{
					/* irregular */
					r = new CIrregularBitmap(P, NumFrames, bitmask);
					int c = 0;
					while(1)
					{
						char *TStr = strdup(Token.Text);
						ExpectToken(TK_COMMA);
						unsigned int Delay;
						GetInteger(Delay);

						((CIrregularBitmap *)r)->SetFrameImage(c, TStr, Delay);
						free(TStr);

						c++;
						if(c == NumFrames) break;
						ExpectToken(TK_COMMA);
						ExpectToken(TK_STRING);
					}
					
				}
				break;
			}
			
			ExpectToken(TK_CLOSEBRACE);
			return r;
		}
		
		case TK_STRING:
		return P->GetResource(Token.Text, bitmask);
		
		default:
			Error = 1;
		return NULL;
	}
}

void Screen::Read()
{
	GetIntegerComma(Id);
	GetStringComma(ScreenName);
	
	GetFloatComma(R);
	GetFloatComma(G);
	GetFloatComma(B);
	
	GetStringComma(ForegroundMap);
	GetStringComma(BackgroundMap);
	GetFloatComma(BackgroundMultiple);
	GetFloatComma(BackScrollX);
	GetFloat(BackScrollY);

	for(int c = 0; c < 4; c++)
	{
		ExpectToken(TK_COMMA);
		GetInteger(ScreenRefs[c]);
	}
}

void ScreenLink::Read()
{
	GetIntegerComma(ScreenRefs[0]);
	GetIntegerComma(ScreenRefs[1]);
	GetFloatComma(AddX);
	GetFloat(AddY);
}

void MapPatch::Read()
{
	GetIntegerComma(Id);
	GetIntegerComma(ScreenRef);
	GetIntegerComma(x);
	GetIntegerComma(y);
	GetString(Filename);
}

void CMap::LoadInitialStats()
{
	ExpectToken(TK_OPENBRACE);
	GetIntegerComma(StartScreen);
	GetFloatComma(StartX);
	GetFloat(StartY);
	ExpectToken(TK_CLOSEBRACE);
}

void DizzyPatch::Read()
{
	GetIntegerComma(Id);

	GetIntegerComma(ScreenRef);
	GetFloatComma(NewX);
	GetFloatComma(NewY);

	GetIntegerComma(Jumping);
	GetInteger(Direction);
}

void ObjectDefinition::Read()
{
	GetIntegerComma(Id);

	GetIntegerComma(JoinFlag);
	GetIntegerComma(CollisionMode);
	GetStringComma(Name);

	char *MapName;
	GetStringComma(MapName);
	if(CollisionMode == 3)
	{
		Tilemap = new CScreenHandler(false);
		Tilemap->Open(MapName);
	}
	free(MapName);

	Image = GetImage(BitmapManager, ((CollisionMode == 1) || (CollisionMode == 2)) ? true : false);
}

void Object::Node::Read()
{
	GetFloatComma(x);
	GetFloatComma(y);
	GetIntegerComma(MirrorMode);
	GetInteger(Time);
}

void Object::Read(ObjectDefinition *DefList)
{
	GetIntegerComma(Id);

	GetIntegerComma(ScreenRef);

	LoadPath();

	/* make list circular */
	CPtr = Path;
	Node *P = Path;
	while(P->Next) P = P->Next;
	P->Next = Path;
	
	BaseTime = 0;

	TotalTime = 0;
	P = Path;
	do
	{
		TotalTime += P->Time;
		P = P->Next;
	}
	while(P != Path);

	ExpectToken(TK_COMMA);
	GetIntegerComma(BehindFlag);
	
	int ObjectNum;
	GetInteger(ObjectNum);
	
	if(ObjectNum)
	{
		Definition = DefList;
		while(Definition->Id != ObjectNum)
			Definition = Definition->Next;
	}
	else
	{
		Definition = NULL;
		Bounder.x1 = Path->x;
		Bounder.y1 = Path->y;
		Bounder.x2 = Path->Next->x;
		Bounder.y2 = Path->Next->y;
		FreePath();
	}
}

void ObjectPatch::Read()
{
	GetIntegerComma(Id);
	GetIntegerComma(ObjectRef);
	GetIntegerComma(ScreenRef);
	GetInteger(ObjectSubstitution);
}

void Event::Read()
{
	GetIntegerComma(ObjectRefs[0]);
	GetIntegerComma(ObjectRefs[1]);
	Key = GetEventKey(ObjectRefs[0], ObjectRefs[1]);

	int ActionEvent;
	GetIntegerComma(ActionEvent);
	Type = ActionEvent ? ET_ACTION : ET_WALKOVER;
	GetIntegerComma(ObjectUseFlag);
	GetIntegerComma(OneTimeFlag);

	GetIntegerComma(MapPatchRef);
	GetIntegerComma(ObjectPatchRef);
	GetIntegerComma(DizzyPatchRef);
	GetString(DialogueSource);
	if(!strlen(DialogueSource)) {free(DialogueSource); DialogueSource = NULL; }

	/* calculate key */
}

void CMap::LoadTiles()
{
	bool Flags[4];

	ExpectToken(TK_OPENBRACE);
	ExpectToken(TK_OPENBRACE);
	while(1)
	{
		// read a new screen
		Uint8 ID;
		ExpectToken(TK_STRING);
		ID = Token.Text[0];
		ExpectToken(TK_COMMA);

		int c = 4;
		while(c--)
		{
			ExpectToken(TK_NUMBER);
			Flags[c] = Token.IQuantity ? true : false;
			ExpectToken(TK_COMMA);
		}

		CurrentScreen->InitTile(ID, GetImage(BitmapManager), Flags[3], Flags[2], Flags[1], Flags[0]);

		// decide whether another exists
		ExpectToken(TK_CLOSEBRACE);

		GetToken();
		if(Token.Type == TK_CLOSEBRACE)
			return;
	}
}

bool CDialogueHandler::Open(char *name)
{
	Error = 0;
	input = host_fopen(name, "rt");
	if(!input) return false;
	ExpectToken(TK_OPENBRACE);
	ExpectToken(TK_OPENBRACE);

	CDialogueHandler *DPtr = this;
	if(Error) return false;
	while(1)
	{
		GetFloatComma(DPtr->r);
		GetFloatComma(DPtr->g);
		GetFloatComma(DPtr->b);
		GetString(DPtr->Text);

		ExpectToken(TK_CLOSEBRACE);
		GetToken();
		if(Token.Type == TK_CLOSEBRACE)
			return true;

		DPtr->Next = new CDialogueHandler;
		DPtr = DPtr->Next;
	}
}

#undef GetInteger
#undef GetIntegerComma

#undef GetFloat
#undef GetFloatComma

#undef GetString
#undef GetStringComma

#define ReadListEx(type, name, arg)\
	type **Top = &name;\
	ExpectToken(TK_OPENBRACE);\
	while(1)\
	{\
		GetToken();\
		if(Token.Type == TK_CLOSEBRACE)\
			break;\
		if(Token.Type != TK_OPENBRACE)\
		{\
			Error = 1;\
			break;\
		}\
		*Top = new type;\
		(*Top)->Read(arg);\
		Top = &(*Top)->Next;\
\
		ExpectToken(TK_CLOSEBRACE);\
	}

#define ReadList(a, b) ReadListEx(a, b, /* */)

void CMap::LoadScreenList()			{ ReadList(Screen, ScreenList); }
void CMap::LoadScreenLinkList()		{ ReadList(ScreenLink, ScreenLinkList); }
void CMap::LoadMapPatches()			{ ReadList(MapPatch, MapPatchList); }
void CMap::LoadDizzyPatches()		{ ReadList(DizzyPatch, DizzyPatchList); }
void CMap::LoadObjectDefinitions()	{ ReadList(ObjectDefinition, ObjectDefinitionList); }
void CMap::LoadObjects()			{ ReadListEx(Object, ObjectList, ObjectDefinitionList); }
void CMap::LoadObjectPatches()		{ ReadList(ObjectPatch, ObjectPatchList); }
void Object::LoadPath()				{ ReadList(Node, Path); }
void CMap::LoadEvents()
{
	Event *TempList;
	ReadList(Event, TempList);
	
	while(TempList)
	{
		Event *Next = TempList->Next;
		TempList->Next = EventList[TempList->Key];
		EventList[TempList->Key] = TempList;

		TempList = Next;
	}
}

#undef ReadList

bool CMap::Open(char *name)
{
	Error = 0;
	if(!(input = host_fopen(name, "rt")))
		return false;

	if(!CurrentScreen)
		CurrentScreen = new CScreenHandler;

	CurrentScreen->FlushTiles();

	LoadTiles();
	LoadScreenList();
	LoadScreenLinkList();

	LoadObjectDefinitions();
	LoadObjects();

	LoadEvents();

	LoadMapPatches();
	LoadDizzyPatches();
	LoadObjectPatches();

	LoadInitialStats();

	fclose(input);
	return true;
}
