#include "Map.h"
#include <stdlib.h>

#include "DizzyState.h"
#include "TimeKeeper.h"
#include "Depths.h"

void CMap::GetInitialScreen(CDizzy &player)
{
	player.x = StartX; player.y = StartY;
	player.Screen = GetScreen(StartScreen);
}

void CMap::ChangeScreen(int Direction, CDizzy &player)
{
	int Next = ScreenPtr->ScreenRefs[Direction];
	GetScreenLink(ScreenPtr->Id, Next);

	if(ScreenLinkPtr->ScreenRefs[0] == ScreenPtr->Id)
	{
		player.x += ScreenLinkPtr->AddX;
		player.y += ScreenLinkPtr->AddY;
	}
	else
	{
		player.x -= ScreenLinkPtr->AddX;
		player.y -= ScreenLinkPtr->AddY;
	}

	player.Screen = GetScreen(Next);
	player.Screen->FinishTransition(Direction, player.x, player.y);
}

Event *CMap::GetEvent(int Id1, int Id2, EventTypes category)
{
	Uint16 Key = GetEventKey(Id1, Id2);
	Event *S = EventList[Key];
	while(S)
	{
		if(S->Type == category)
		{
			if(
				(S->ObjectRefs[0] == Id1 && S->ObjectRefs[1] == Id2) ||
				(S->ObjectRefs[1] == Id1 && S->ObjectRefs[0] == Id2)
			)
			{
				return S;
			}
		}
		S = S->Next;
	}
	return NULL;
}

CScreenHandler *CMap::GetScreen(int id)
{
	/* find ScreenPtr */
	ScreenPtr = ScreenList;
	while(ScreenPtr->Id != id)
		ScreenPtr = ScreenPtr->Next;

	MakeScreenLists();

	/* install tilemap + any existing patches */
	glClearColor(ScreenPtr->R, ScreenPtr->G, ScreenPtr->B, 0);
	CurrentScreen->Open(ScreenPtr->ForegroundMap, ScreenPtr->BackgroundMap, ScreenPtr->BackgroundMultiple, ScreenPtr->BackScrollX, ScreenPtr->BackScrollY);
	MapPatch *MP = ScreenPtr->Patches;
	while(MP)
	{
		CurrentScreen->AddPatch(MP->Filename, MP->x, MP->y);
		MP = MP->ScreenNext;
	}

	/* return */
	CurrentScreen->SetName(ScreenPtr->ScreenName);
	return CurrentScreen;
}

void CMap::MakeScreenLists()
{
	/* thread Object list */
	ScreenObjectList = NULL;
	Object *CP = ObjectList;
	while(CP)
	{
		if(CP->ScreenRef == ScreenPtr->Id)
		{
			CP->ScreenNext = ScreenObjectList;
			ScreenObjectList = CP;
		}
		CP = CP->Next;
	}
}

void CMap::GetScreenLink(int id1, int id2)
{
	ScreenLinkPtr = ScreenLinkList;
	while(
		! 
			(
				((ScreenLinkPtr->ScreenRefs[0] == id1) && (ScreenLinkPtr->ScreenRefs[1] == id2)) ||
				((ScreenLinkPtr->ScreenRefs[1] == id1) && (ScreenLinkPtr->ScreenRefs[0] == id2))
			)
	)
		ScreenLinkPtr = ScreenLinkPtr->Next;
}


void Object::CalculateBounder()
{
	float w, h;
	w = Definition->Image->GetW()*0.5f;
	h = Definition->Image->GetW()*0.5f;
	Bounder.x1 = CurrentX - w; Bounder.x2 = CurrentX + w;
	Bounder.y1 = CurrentY - h; Bounder.y2 = CurrentY + h;
}

void Object::CalculateCorrectXY()
{
	if(Path && TimeKeeper.Quantums != CurrentQuantum)
	{
		CurrentQuantum = TimeKeeper.Quantums;

		unsigned int TimeOffset = (TimeKeeper.Quantums - BaseTime)%TotalTime;
		BaseTime += (TimeKeeper.Quantums - BaseTime)-TimeOffset;
		while(TimeOffset >= CPtr->Time)
		{
			BaseTime += CPtr->Time;
			TimeOffset -= CPtr->Time;
			CPtr = CPtr->Next;
		}

		float Ratio = (float)TimeOffset / (float)CPtr->Time;
		CurrentX = (1.0f - Ratio)*CPtr->x + Ratio*CPtr->Next->x;
		CurrentY = (1.0f - Ratio)*CPtr->y + Ratio*CPtr->Next->y;
		CurrentMirrorMode = CPtr->MirrorMode;

		CalculateBounder();
	}
}

void CMap::AddObjects()
{
	glColor3f(1, 1, 1);

	/* add Objects */
	Object *C = ScreenObjectList;
	while(C)
	{
		if(C->Definition)
		{
			/* calculate x, y */
			C->CalculateCorrectXY();
			glPushMatrix();
				if(C->Definition->JoinFlag)
				{
					glDisable(GL_TEXTURE_2D);
						glBegin(GL_LINES);
							glVertex3f(C->Path->x, C->Path->y, C->BehindFlag ? DEPTH_BACKGROUND_ObjectS : DEPTH_FOREGROUND_ObjectS);
							glVertex3f(C->CurrentX, C->CurrentY, C->BehindFlag ? DEPTH_BACKGROUND_ObjectS : DEPTH_FOREGROUND_ObjectS);
						glEnd();
					glEnable(GL_TEXTURE_2D);
				}

				glTranslatef(C->CurrentX, C->CurrentY, 0);
				switch(C->CurrentMirrorMode)
				{
					default: break;
					case 1: glScalef(-1, 1, 1); break;
					case 2: glScalef(1, -1, 1); break;
					case 3: glScalef(-1, -1, 1); break;
				}
				C->Definition->Image->Draw( C->BehindFlag ? DEPTH_BACKGROUND_ObjectS : DEPTH_FOREGROUND_ObjectS);
			glPopMatrix();
		}
		C = C->ScreenNext;
	}
}

void CMap::SortObjects()
{
	/* sorts ScreenObjectList from left to right, according to bounding box x1 */
	
	/*

		Uses insertion sort. Since most of the time the list will come already
		very nearly sorted, this is the most efficient one

	*/

	if(!ScreenObjectList || !ScreenObjectList->ScreenNext) return; // cannot sort a 0 or 1 item list!

	Object **C = &ScreenObjectList;
	while((*C)->ScreenNext)
	{
		if((*C)->ScreenNext->Bounder.x1 < (*C)->Bounder.x1)
		{
			/* unlink */
			Object *MotionNode = (*C)->ScreenNext;
			(*C)->ScreenNext = (*C)->ScreenNext->ScreenNext;

			/* find proper insertion place */
			Object **C2 = &ScreenObjectList;
			while( (*C2)->Bounder.x1 < MotionNode->Bounder.x1)
				C2 = &(*C2)->ScreenNext;

			/* link in */
			MotionNode->ScreenNext = *C2;
			*C2 = MotionNode;
		}
		else
			C = &(*C)->ScreenNext;
	}
}

#define Queue(E)	\
				E->QueueNext = player.TriggeredEvent;\
				player.TriggeredEvent = E;

int CMap::ProcessPosition(CDizzy &player, float &escx, float &escy, float &energy)
{
	int Changes = 0;
	bool Pickup = false;
	bool ListsDirty = false;

	/* clear up current events queue if necessary */
	if(player.TriggeredEvent)
	{
		ListsDirty |= EnactEvent(player.TriggeredEvent, player);
		player.TriggeredEvent = player.TriggeredEvent->QueueNext;
	}

	/* update object positions, sort list */
	Object *C = ScreenObjectList;
	while(C)
	{
		C->CalculateCorrectXY();
		C = C->ScreenNext;
	}
	SortObjects();

	/* now go searching for new events - first as against player */
	C = ScreenObjectList;
	while(C && C->Bounder.x2 <= player.Bounder.x1)
		C = C->ScreenNext;
	while(C && C->Bounder.x1 <= player.Bounder.x2)
	{
		/* do stuff triggered by Dizzy */
		if(player.Bounder.Overlaps(C->Bounder))
		{
			/* queue any mere overlap events */
			Event *NewE;
			if(NewE = GetEvent(0, C->Id, ET_WALKOVER))
			{
				Queue(NewE);
			}

			/* queue any action events if relevant */
			if(player.Interacting && (NewE = GetEvent(0, C->Id, ET_ACTION)))
			{
				Queue(NewE);
			}

			/* check for overlap and action events from carried objects */
			int c = 5;
			while(c--)
			{
				if(player.Objects[c])
				{
					if(NewE = GetEvent(player.Objects[c]->Id, C->Id, player.Interacting ? ET_ACTION : ET_WALKOVER))
					{
						Queue(NewE);
					}
				}
			}

			/* consider interactions with the player */
			if(C->Definition)
			switch(C->Definition->CollisionMode)
			{
				default: break;	/* default = 0 = no collision */
				case 1:
				case 2:	/* painful or bouncing collision */
				{
					int PMask;
					if(C->Definition->Image->CheckCollision( player.GetImage(PMask), player.x - C->CurrentX, player.y - C->CurrentY, C->CurrentMirrorMode, PMask))
					{
						if(C->Definition->CollisionMode == 1)
							energy--;
						else
						{
							player.Direction = (player.x > C->CurrentX) ? 1 : -1;
							player.StartJump();
						}
					}
				}
				break;
				case 3: /* tilemap collision */
				{
					float OffsetX, OffsetY;
					OffsetX = player.x - C->CurrentX;
					OffsetY = player.y - C->CurrentY;
					float AddX = 0, AddY = 0;

					float BackupX = player.x, BackupY = player.y;
					float ymult = 1.0f, xmult = 1.0f;

					switch(C->CurrentMirrorMode)
					{
						default: break;
						case 1: xmult = -1; break;
						case 2: ymult = -1; break;
						case 3: xmult = ymult = -1; break;
					}
					player.x = OffsetX*xmult + (C->Definition->Image->GetW()*0.5f);
					player.y = OffsetY*ymult + (C->Definition->Image->GetH()*0.5f);
					player.yadd = ymult*player.yadd;
					C->Definition->Tilemap->AddEscapeVector(player, AddX, AddY, energy);
					escx += xmult*AddX;
					escy += ymult*AddY;
					player.yadd = ymult*player.yadd;
					player.x = BackupX;
					player.y = BackupY;
				}
				break;
				case 4: /* collectable */
					if(player.Interacting)
					{
						/* if carrying too many objects, put one down */
						if(player.Objects[4])
						{
							PutObject(player.Objects[0], player);
							player.ShiftObjects(0);
						}
						/* pick this object up */
						int c = 4;
						while(1)
						{
							if(!c) { player.Objects[0] = C; break; }
							if(!player.Objects[c-1])
								c--;
							else
							{
								player.Objects[c] = C;
								break;
							}
						}
						C->ScreenRef = 0;
						Pickup = ListsDirty = true;
						Changes |= CMC_OBJECT;
					}
				break;
			}
		}

		C = C->ScreenNext;
	}
		
	/* now as against other objects */
	C = ScreenObjectList;
	while(C)
	{
		Object *C2 = C->ScreenNext;
		while(C2 && C2->Bounder.x1 <= C->Bounder.x2)
		{
			if(C2->Bounder.Overlaps(C->Bounder))
			{
				Event *NewE;
				if(NewE = GetEvent(C2->Id, C->Id, ET_WALKOVER))
				{
					Queue(NewE);
				}
			}
			C2 = C2->ScreenNext;
		}

		C = C->ScreenNext;
	}

	/* do events, etc! */
	if(player.TriggeredEvent)
	{
		/* remove any events that pair two non-Dizzy objects and share either */
		Event **EPtr = &player.TriggeredEvent, **E2Ptr;
		while(*EPtr)
		{
			int TestId1 = (*EPtr)->ObjectRefs[0], TestId2 = (*EPtr)->ObjectRefs[1];
			E2Ptr = &(*EPtr)->QueueNext;
			while(*E2Ptr)
			{
				if( 
					((*E2Ptr)->ObjectRefs[0] == TestId1 && (*E2Ptr)->ObjectRefs[1] && TestId1) ||
					((*E2Ptr)->ObjectRefs[1] == TestId1 && (*E2Ptr)->ObjectRefs[0] && TestId1) ||
					((*E2Ptr)->ObjectRefs[0] == TestId2 && (*E2Ptr)->ObjectRefs[1] && TestId2) ||
					((*E2Ptr)->ObjectRefs[1] == TestId2 && (*E2Ptr)->ObjectRefs[0] && TestId2)
				)
				{
					*E2Ptr = (*E2Ptr)->QueueNext;
				}
				else
					E2Ptr = &(*E2Ptr)->QueueNext;
			}

			EPtr = &(*EPtr)->QueueNext;
		}
	
		/* do all events that don't generate a dialogue */
		EPtr = &player.TriggeredEvent;
		while(*EPtr)
		{
			if(! (*EPtr)->DialogueSource)
			{
				ListsDirty |= EnactEvent(*EPtr, player);
				*EPtr = (*EPtr)->QueueNext;
			}
			else
				EPtr = &(*EPtr)->QueueNext;
		}

		/* trigger first dialogue if any exist */
		if(player.TriggeredEvent)
		{
			player.CurrentDialogue = new CDialogueHandler;
			player.CurrentDialogue->Open(player.TriggeredEvent->DialogueSource);
			Changes |= CMC_DIALOGUE;
		}
	}
	else
		if(!Pickup && player.Interacting && player.Objects[0])
		{
			/* just put an object down */
			PutObject(player.Objects[0], player);
			player.ShiftObjects(0);
		}

	if(ListsDirty)
		MakeScreenLists();

	/* get escape vector for current tilemap */
	CurrentScreen->AddEscapeVector(player, escx, escy, energy);

	/* check whether we've moved beyond the current tilemap */
	int Direction = CurrentScreen->QueryScreenChange(player.x, player.y);
	if(Direction >= 0) ChangeScreen(Direction, player);

	/* return flags denoting what status area affecting changes have happened */
	return Changes;
}

#undef Queue

void CMap::PutObject(Object *obj, CDizzy &player)
{
	obj->FreePath();
	int t;
	obj->CurrentX = player.x; obj->CurrentY = player.y + (player.GetImage(t)->GetH() - obj->Definition->Image->GetH()) * 0.5f;
	obj->CalculateBounder();

	obj->ScreenRef = ScreenPtr->Id;
	obj->ScreenNext = ScreenObjectList;
	ScreenObjectList = obj;
}

bool CMap::EnactEvent(Event *e, CDizzy &Dizzy)
{
	bool ListsDirty = false;

	if(e->MapPatchRef)
	{
		// find map patch
		MapPatch *MP = MapPatchList;
		while(MP)
		{
			if(MP->Id == e->MapPatchRef)
			{
				// thread map patch into appropriate place
				Screen *SP = ScreenList;
				while(SP->Id != MP->ScreenRef)
					SP = SP->Next;

				MP->ScreenNext = SP->Patches;
				SP->Patches = MP;
				if(SP == ScreenPtr)
				{
					CurrentScreen->AddPatch(MP->Filename, MP->x, MP->y);
				}
			}
			MP = MP->Next;
		}
	}

	if(e->ObjectPatchRef)
	{
		ObjectPatch *CP = ObjectPatchList;
		while(CP)
		{
			if(CP->Id == e->ObjectPatchRef)
			{
				Object *CR = ObjectList;
				while(CR)
				{
					if(CR->Id == CP->ObjectRef)
					{
						CR->ScreenRef = CP->ScreenRef;
						ListsDirty = true;
					}
					CR = CR->Next;
				}
			}
			CP = CP->Next;
		}
	}

	if(e->DizzyPatchRef)
	{
		// only expect one of these
		DizzyPatch *DP = DizzyPatchList;
		while(DP->Id != e->DizzyPatchRef)
			DP = DP->Next;

		if(DP->Jumping && !Dizzy.Jumping)
		{
			Dizzy.Direction = DP->Direction;
			Dizzy.StartJump();
		}

		// is Dizzy now changing screen?
		if(DP->ScreenRef)
		{
			Dizzy.x = DP->NewX;
			Dizzy.y = DP->NewY;
			Dizzy.Screen = GetScreen(DP->ScreenRef);
			ListsDirty = false;
		}
	}

	// disable any dialogue associated with this event
	if(e->DialogueSource)
	{
		free(e->DialogueSource);
		e->DialogueSource = NULL;
	}

	// find out if we've just eaten an object
	if(e->ObjectUseFlag)
	{
		int c = 5;
		while(c--)
		{
			if(Dizzy.Objects[c] && (Dizzy.Objects[c]->Id == e->ObjectRefs[0] || Dizzy.Objects[c]->Id == e->ObjectRefs[1]))
			{
				Dizzy.ShiftObjects(c);
				break;
			}
		}
	}

	// disable this event if it is one time only
	if(e->OneTimeFlag)
	{
		/* quick fix! */
		e->ObjectRefs[0] = e->ObjectRefs[1] = 0;
	}

	return ListsDirty;
}

void CDialogueHandler::PickLocation(CTextWriter *Writer)
{
	Position.y1 = 42;
	Position.x1 = 30;

	Position.x2 = Position.x1 + Writer->GetLength(Text) + 4;
	if(Position.x2 > 286)
	{
		Position.x2 = 286;
		Position.y2 = Position.y1 + Writer->GetHeight(Position.x1+2, Position.x2-2, Text) + 4;
	}
	else
	{
		float HWidth = (Position.x2 - Position.x1) * 0.5f;
		Position.y2 = Position.y1 + 10;
		Position.x1 = 160 - HWidth;
		Position.x2 = 160 + HWidth;
	}
}

bool FloatingRect::Overlaps(FloatingRect &o)
{
	if(x2 < o.x1) return false;
	if(x1 > o.x2) return false;
	if(y2 < o.y1) return false;
	if(y1 > o.y2) return false;
	return true;
}
