#include "BitmapManager.h"
#include "SDL/SDL_image.h"
#include "Archive.h"
#include "HostMachine.h"
#include <string.h>
#include "BitmapRepository.h"

CBitmapParent *BitmapManager;
GLuint CBitmapParent::CTexNum;

CBitmapParent::CBitmapParent(int w, int h)
{
	CTexNum = TexNum = 0;
	CX = CY = LineHeight = 0;
	TexW = w; TexH = h;
	Bitmap = NULL;
	BitmapRepository.Add(this);
}

CBitmapParent::~CBitmapParent()
{
	BitmapRepository.Remove(this);
	if(TexNum)
	{
		glDeleteTextures(1, &TexNum);
		TexNum = 0;
	}
	delete[] Bitmap;
}

void CBitmapParent::SetParameters()
{
	glGenTextures(1, &TexNum);
	glBindTexture(GL_TEXTURE_2D, TexNum);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);
}

void CBitmapParent::Activate()
{
	if(Bitmap)
	{
		SetParameters();
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, TexW, TexH, 0, GL_RGBA, GL_UNSIGNED_BYTE, Bitmap);
		delete[] Bitmap;
		Bitmap = NULL;
	}
	if(CTexNum != TexNum) glBindTexture(GL_TEXTURE_2D, CTexNum = TexNum);
}

void CBitmapParent::Backup()
{
	if(!Bitmap)
	{
		Activate();
		Bitmap = new Uint32[TexW*TexH];
		glGetTexImage(GL_TEXTURE_2D, 0, GL_RGBA, GL_UNSIGNED_BYTE, Bitmap);
	}
}

CBitmap *CBitmapParent::GetResource(char *name, bool generatemask)
{
	/* attempt to open resource name */
	SDL_RWops *OpPtr;
	SDL_Surface *NewBitmap = IMG_Load_RW(OpPtr = FileStore.GetRWops(name), 0);
	if(!NewBitmap) return NULL;

	/* check if we need to allocate the bitmap, etc */
	if(!TexNum)
	{
		SetParameters();

		Uint32 *tData = new Uint32[TexW*TexH];
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, TexW, TexH, 0, GL_RGBA, GL_UNSIGNED_BYTE, tData);
		delete[] tData;
	}
	else
		Activate();

	/* resource successfully opened, so find a spot for it */
	CBitmap *r = new CBitmap;
	Uint32 *NewData = new Uint32[NewBitmap->w * NewBitmap->h];

	if((CX+NewBitmap->w) >= TexW)
	{
		CX = 0;
		CY += LineHeight;
		LineHeight = 0;
	}
	r->x1 = (float)CX / (float)TexW;
	r->y1 = (float)CY / (float)TexH;
	r->x2 = (float)(CX+NewBitmap->w) / (float)TexW;
	r->y2 = (float)(CY+NewBitmap->h) / (float)TexH;
	r->w = (float)NewBitmap->w;
	r->h = (float)NewBitmap->h;
	r->BParent = this;
	if(NewBitmap->h > LineHeight) LineHeight = NewBitmap->h;

	/* and copy in pixels, making bitmask if interesting */
	if(generatemask)
	{
		r->Mask.TrueW = NewBitmap->w;
		r->Mask.BitW = (NewBitmap->w+31) >> 5;
		r->Mask.H = NewBitmap->h;
		
		r->Mask.ForwardMask = new Uint32[r->Mask.BitW * NewBitmap->h];
		r->Mask.BackwardMask = new Uint32[r->Mask.BitW * NewBitmap->h];

		memset(r->Mask.ForwardMask, 0, sizeof(Uint32)*r->Mask.BitW * NewBitmap->h);
		memset(r->Mask.BackwardMask, 0, sizeof(Uint32)*r->Mask.BitW * NewBitmap->h);
	}

	SDL_LockSurface(NewBitmap);
	int x, y;
	for(y = 0; y < NewBitmap->h; y++)
	{
		for(x = 0; x < NewBitmap->w; x++)
		{
			int Colour;
			switch(NewBitmap->format->BitsPerPixel)
			{
				default:
				case 8:
					Colour = ((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + x];
				break;
				case 15:
				case 16:
					Colour = ((Uint16 *)(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch]))[x];
				break;
				case 24:
#if SDL_BYTEORDER == SDL_LIL_ENDIAN
					Colour = 
						(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + (x*3)] << 0) |
						(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + (x*3) + 1] << 8) |
						(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + (x*3) + 2] << 16);
#else
					Colour = 
						(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + (x*3)] << 16) |
						(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + (x*3) + 1] << 8) |
						(((Uint8 *)NewBitmap->pixels)[y*NewBitmap->pitch + (x*3) + 2] << 0);
#endif
				break;
				case 32:
					Uint8 *BasePtr = (Uint8 *)NewBitmap->pixels;
					Uint8 *LinePtr = &BasePtr[y*NewBitmap->pitch];
					Colour = ((Uint32 *)LinePtr)[x];
				break;
			}
			
			Uint8 rd, gn, bl, a;
			SDL_GetRGBA(Colour, NewBitmap->format, &rd, &gn, &bl, &a);
			((Uint8 *)&NewData[ (y*NewBitmap->w) + x])[0] = rd;
			((Uint8 *)&NewData[ (y*NewBitmap->w) + x])[1] = gn;
			((Uint8 *)&NewData[ (y*NewBitmap->w) + x])[2] = bl;

			if((rd == 0xff) && (gn == 0x00) && (bl == 0xff))
				((Uint8 *)&NewData[ (y*NewBitmap->w) + x])[3] = 0;
			else
				((Uint8 *)&NewData[ (y*NewBitmap->w) + x])[3] = a;

			if(generatemask && ((Uint8 *)&NewData[ (y*NewBitmap->w) + x])[3])
			{
				r->Mask.ForwardMask[(y*r->Mask.BitW) + (x >> 5)] |= 1 << (31-(x&31));

				int transx = NewBitmap->w-x;
				r->Mask.BackwardMask[(y*r->Mask.BitW) + (transx >> 5)] |= 1 << (31-(transx&31));
			}
		}
	}
	SDL_UnlockSurface(NewBitmap);
	glTexSubImage2D(GL_TEXTURE_2D, 0, CX, CY, NewBitmap->w, NewBitmap->h, GL_RGBA, GL_UNSIGNED_BYTE, NewData);
	delete[] NewData;

	CX += NewBitmap->w;

	/* test print! */
/*	if(generatemask)
	{
		for(int y = 0; y < r->Mask.H; y++)
		{
			for(int x = 0; x < r->Mask.BitW; x++)
			{
				Uint32 Value = r->Mask.Mask[y*r->Mask.BitW + x];
				for(int c = 0; c < 32; c++)
				{
					printf("%c", (Value&0x80000000) ? '.' : ' ');
					Value <<= 1;
				}
			}
			printf("\n");
		}
	}*/

	/* free memory and return */
	SDL_FreeSurface(NewBitmap);
	FileStore.ReturnRWops(OpPtr);
	return r;
}

float CBitmap::GetW()
{
	return w;
}

float CBitmap::GetH()
{
	return h;
}

void CBitmap::Draw(float Depth, float x, float y)
{
	BParent->Activate();
	
	/* white test */
//	GLfloat AmbientLight[4] = {1, 1, 0, 1};
//	glLightfv(GL_LIGHT0, GL_AMBIENT, AmbientLight);
	
//	GLfloat Material[4] = {1, 0.1, 1, 1};
//	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, Material);
//	glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION
//	glEnable(GL_LIGHT0);
//	glEnable(GL_LIGHTING);

	/*GLfloat Colour[4] = {1, 1, 1, 1};
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND);
	glTexEnvfv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_COLOR, Colour);
	glColor3f(1, 1, 1, 0.5f);*/

	glBegin(GL_QUADS);
		glTexCoord2f(x1, y1); glVertex3f(x - w*0.5f, y - h*0.5f, Depth);
		glTexCoord2f(x2, y1); glVertex3f(x + w*0.5f, y - h*0.5f, Depth);
		glTexCoord2f(x2, y2); glVertex3f(x + w*0.5f, y + h*0.5f, Depth);
		glTexCoord2f(x1, y2); glVertex3f(x - w*0.5f, y + h*0.5f, Depth);
	glEnd();
}

CBitmap::~CBitmap() {}

bool CBitmap::CheckCollision(CBitmap *other, float offx, float offy, int DirFlags1, int DirFlags2)
{
	/* do separating planes test */
	float x1, y1, x2, y2;
	x1 = offx - other->GetW()*0.5f;
	x2 = offx + other->GetW()*0.5f;
	if(x1 > w*0.5f) return false;
	if(x2 < -w*0.5f) return false;

	y1 = offy - other->GetH()*0.5f;
	y2 = offy + other->GetH()*0.5f;
	if(y1 > h*0.5f) return false;
	if(y2 < -h*0.5f) return false;

	/* get opposing bit mask */
	BitMask *A, *B;

	A = GetBitMask();
	B = other->GetBitMask();

	/* figure out which is the leftmost bitmap, make it 'A' */
	if(offx < 0)
	{
		BitMask *T = A;
		A = B; B = T;
		int DFS = DirFlags1;
		DirFlags1 = DirFlags2; DirFlags2 = DFS;
		offx = -offx; offy = -offy;
	}

	/* get integer offsets of B from A */
	int IntegerOffX = (int)(offx - (float)(B->TrueW - A->TrueW)*0.5f);
	int IntegerOffY = (int)(offy - (float)(B->H - A->H)*0.5f);

	/* iterate through overlapping y region */
	int StartY, EndY;
	StartY = (IntegerOffY < 0) ? 0 : IntegerOffY;
	EndY = ((IntegerOffY+B->H) < A->H) ? (IntegerOffY+B->H) : A->H;

	int StartX, EndX;
	StartX = (IntegerOffX < 0) ? 0 : IntegerOffX;
	EndX = ((IntegerOffX+B->TrueW) < A->TrueW) ? (IntegerOffX+B->TrueW) : A->TrueW;

	StartX >>= 5;
	EndX = (EndX+31) >> 5;
	int ShiftMask = IntegerOffX&31;
	int OffX = IntegerOffX >> 5;
	
	Uint32 *AMask, *BMask;
	AMask = (DirFlags1&DIRFLAG_REVERSE_X) ? A->ForwardMask : A->BackwardMask;
	BMask = (DirFlags2&DIRFLAG_REVERSE_X) ? B->ForwardMask : B->BackwardMask;

	for(int iy = StartY; iy < EndY; iy++)
	{
		for(int ix = StartX; ix < EndX; ix++)
		{
			Uint32 AValue = AMask[iy*A->BitW + ix];
			if(AValue)
			{
				Uint32 BValue = BMask[ (iy-IntegerOffY)*B->BitW + (ix-OffX)] >> ShiftMask;
				if(ix-OffX && ShiftMask)
					BValue |= BMask[ (iy-IntegerOffY)*B->BitW + (ix-OffX-1)] << (32-ShiftMask);

				if(AValue&BValue)
					return true;
			}
		}
	}

	return false;
}

CBitmap::BitMask *CBitmap::GetBitMask()
{
	return &Mask;
}
