#include "Archive.h"
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "HostMachine.h"

CArchive FileStore;

#define Read32(t)\
	t = gzgetc(afile);\
	t |= gzgetc(afile) << 8;\
	t |= gzgetc(afile) << 16;\
	t |= gzgetc(afile) << 24

#define Write32(a, t)\
	gzputc(a, t&0xff);\
	gzputc(a, (t >> 8)&0xff);\
	gzputc(a, (t >> 16)&0xff);\
	gzputc(a, (t >> 24)&0xff);
	
#define FILENAME "dizzy.dat"

void CArchive::OpenFile()
{
	afile = host_gzopen(FILENAME, "rb");
	if(afile)
	{
		int c = 64;
		while(c--)
		{
			Read32(Offsets[c]);
			Read32(Counts[c]);
		}
	}
	else
	{
		memset(Offsets, 0, sizeof(unsigned int)*64);
		memset(Counts, 0, sizeof(unsigned int)*64);
	}
}

CArchive::CArchive()
{
	OpenFile();
	Data = NULL;
	Length = 0;
}

CArchive::~CArchive()
{
	if(afile)
		gzclose(afile);

	if(Data)
		delete[] Data;
}

void CArchive::ArchiveFile(char *name)
{
	printf("Archiving %s\n", name);
	int c;
	/* adds 'name' to the file */

	/* determine offset for this letter */
	int Index = toupper(name[0])-32, Offset;
	if(!(Offset = Offsets[Index]))
	{
		for(c = 64; c > Index; c--)
			if(Offsets[c]) Offset = Offsets[Index] = Offsets[c];

		if(!Offset)
		{
			/* set Offset to end of file - find that first! */
			if(!afile)
			{
				Offset = Offsets[Index] = 64*8;
			}
			else
			{
				gzseek(afile, (1 << 20), SEEK_SET);
				Offset = Offsets[Index] = gztell(afile);
			}
		}
	}
	Counts[Index]++;

	/* adjust the offsets for all following letters */
	struct stat fileinfo;
	stat(name, &fileinfo);

	for(c = Index+1; c < 64; c++)
		if(Offsets[c]) Offsets[c] += fileinfo.st_size + 4 + strlen(name) + 1;

	/* create a new file and copy stuff over if necessary */
	gzFile newfile = host_gzopen(afile ? "__tmp__.tmp" : FILENAME, "wb9");
	
	/* write offsets and counts */
	c = 64;
	while(c--)
	{
		Write32(newfile, Offsets[c]);
		Write32(newfile, Counts[c]);
	}

	if(afile)
	{
		/* seek same position in existing file */
		gzseek(afile, 64*8, SEEK_SET);

		/* copy up to offset in a very dense fashion */
		c = Offsets[Index] - 64*8;
		while(c--)
		{
			Uint8 td = gzgetc(afile);
			gzputc(newfile, td);
		}
	}

	/* write data pertaining to new file */
	Write32(newfile, fileinfo.st_size);
	char *cptr = name;
	while(*cptr)
	{
		gzputc(newfile, *cptr);
		cptr++;
	}
	gzputc(newfile, 0);

	/* copy new file */
	FILE *newf = host_fopen(name, "rb");
	c = fileinfo.st_size;
	while(c--)
	{
		Uint8 td = fgetc(newf);
		gzputc(newfile, td);
	}
	fclose(newf);

	if(afile)
	{
		int Test = gztell(afile);
		int c= 0;
		printf("Copying from %d onwards\n", Test);
		/* write rest of data from original file */
		while(1)
		{
			Uint8 td = gzgetc(afile);
			gzputc(newfile, td);c++;
			if(gzeof(afile)) break;
		}
		printf("%d bytes\n", c);
		gzclose(afile);
	}

	/* close both files */
	int Test = gztell(newfile);
	printf("New size: %d\n", Test);
	gzclose(newfile);

	/* erase original, rename new copy */
	if(afile)
	{
		unlink(FILENAME);
		rename("__tmp__.tmp", FILENAME);
	}

	/* reopen */
	OpenFile();
}

bool CArchive::Find(char *name)
{
	int Index = toupper(name[0]) - 32, Total;
	if(!(Total = Counts[Index])) return false;

//	printf("seeking %d\n", Offsets[Index]);
	gzseek(afile, Offsets[Index], SEEK_SET);

	while(Total--)
	{
		/* read length of file */
//		printf("Reading length from %d\n", gztell(afile));
		Read32(Length);

		/* read filename of file */
		char Name[1024], *NPtr = Name;
		while(1)
		{
			*NPtr = gzgetc(afile);
			if(!(*NPtr)) break;
			NPtr++;
		}
		
		/* if we have a hit, return */
		if(!strcmp(name, Name)) {printf("%s found\n", name); return true;}

		/* otherwise, advance */
		gzseek(afile, Length, SEEK_CUR);
	}

	/* oh dear, not found */
	return false;
}

SDL_RWops *CArchive::GetRWops(char *name)
{
	if(!Find(name))
	{
		char TrueName[2048];
		sprintf(TrueName, "%s%c%s", GetResourcePath(), SeparatorChar(), name);
		return SDL_RWFromFile(TrueName, "rb");
//		ArchiveFile(name);
//		Find(name);
	}
	
	/* read whole file in, make RWops */
	Data = new Uint8[Length];
	gzread(afile, Data, Length);
	
	return SDL_RWFromMem(Data, Length);
}

void CArchive::ReturnRWops(SDL_RWops *ops)
{
	delete[] Data; Data = NULL;
	SDL_FreeRW(ops);
}

gzFile CArchive::GetGZFilePtr(char *name)
{
	if(!Find(name))
	{
		return host_gzopen(name, "rb");
//		ArchiveFile(name);
//		Find(name);
	}

	return afile;
}

void CArchive::ReturnGZFilePtr(gzFile f)
{
	if(f != afile)
		gzclose(f);
}
